function [check_data] = check_centroids(ref_ch,varargin)
%Takes the data structure from find_centroid_smart or multi and runs a
%sanity check on whether the centroids are out of line.  Uses a reference
%channel, e.g., PSD95
%Synatax:   [] = check_centroids('psd95','data',centroid_data,'dirname','c:\...','filename','centroid_data.mat','std_away',2,'method','median')
%Input:     'dirname' = directory where structure file resides
%           'filename' = the name of the file holding the centroid structure
%               don't forget the filesep after at the end of the directory.
%           ref_ch = the reference channel to use to compare the other
%               channels.  I suggest PSD95.
%           'data' = The centroid data structure, if provided then filename
%               and dirname are not necessary
%           'std_away' = How many standard deviations away do we consider
%               the centroid number to be too far.  Default = 2
%           'method' = What method of data summary do we use: median, mean
%               or mode.  Default = median
%Output:    check_data = Output of the analysis

[cdata,method,std_away,dirname] = parse(varargin);

%now run through the directory structure.
%lets initiate some parameters
limit = [0.03,0.18]; %hard limit on thresholds
norm = cdata.norm;
object = cdata.object;
zfilter = cdata.zfilter;
type = cdata.type;
type2 = cdata.type2;
check_data = {'experiment','animal','region','channel','orig_thresh','new_thresh','orig_centroid','new_centroid',...
    'ref_stat','ref_std','upper_bound','lower_bound','in_dir','out_dir'};
%first get the number of fields
animal_types = fieldnames(cdata);  %types of animals
%step through
for i = 1:size(animal_types,1)
    if isstruct(cdata.(animal_types{i}))   %only move forward if this field leads to a structure
        %get the regions
        regions = fieldnames(cdata.(animal_types{i}));
        %now step through the regions
        for j = 1:size(regions,1)
            %first grab the data structure and the channel references
            cdata_tmp = cell2mat(nixemptycell(cdata.(animal_types{i}).(regions{j}).centroid(2:end,2:end)));
            ch_names = cdata.(animal_types{i}).(regions{j}).centroid(1,:);
            exp_names = cdata.(animal_types{i}).(regions{j}).centroid(:,1);
            thresh_tmp = cdata.(animal_types{i}).(regions{j}).threshold(2:end,2:end);
            loc_tmp = cdata.(animal_types{i}).(regions{j}).location(2:end,2:end);
            %now find the reference channel
            for k = 1:size(ch_names,2)
                if ~isempty(strfind(ch_names{k},ref_ch))  %found it
                    ref_idx = k-1;  %here's the reference channel.  The first column is blank
                    break
                end
            end
            %now calculate the ratio across all channels
            %first replicate the reference channel to all channels
            ref_data = repmat(cdata_tmp(:,ref_idx),1,size(cdata_tmp,2));
            ratio_data = cdata_tmp./ref_data;
            %now test each channel
            for l = 1:size(cdata_tmp,2)
                if l~=ref_idx  %skip the reference channel itself
                    %calculate the ratios first ch/ref_ch
                    ref_ratio = ratio_data(:,l);
                    %calculate the reference statistic
                    switch method
                        case 'mean'
                            ref_stat = nanmean(ref_ratio);
                        case 'mode'
                            ref_stat = mode(ref_ratio);
                        case 'median'
                            ref_stat = nanmedian(ref_ratio);
                    end
                    %calculate the standard deviation
                    ref_std = std(ref_ratio);
                    %calculate the reference cutoff
                    ref_cutup = ref_stat+(ref_std*std_away);  %upper bound
                    ref_cutdwn = ref_stat-(ref_std*std_away);   %lower bound
                    %deal with upper bound first
                    out_idx = find(ratio_data(:,l)>=ref_cutup);
                    %now fix this centroid
                    if ~isempty(out_idx)  %skip if there are no out of bounds centroids
                        for m = 1:size(out_idx,2)  %go through each centroid
                            %grab the threshold and the locations now
                            threshold = thresh_tmp{out_idx(m),l};
                            loc_cache = loc_tmp{out_idx(m),l};
                            curr_dir = loc_cache{1};
                            dirout = loc_cache{2};
                            stk = [];  %initialize
                            lim_rev = 0;  %a switch to verify that the limit is reached
                            no_deal = 0;
                            disp([curr_dir,' - old threshold:',num2str(threshold),' with ratio:',num2str(ratio_data(out_idx(m),l))]);  %print current directory to command line
                            while 1  %keep on going until we fix it
                                %the centroid here is high.  So we need to
                                %increase the threshold
                                if isempty(stk)  %first time through and we add first
                                    t_tmp = threshold+0.01;
                                else
                                    if lim_rev
                                        t_tmp = t_tmp-0.01;  %subtract if lim_rev is reached
                                    else
                                        t_tmp = t_tmp+0.01;  %add if lim_rev is not reached
                                    end
                                end
                                if t_tmp>limit(2)  %gone past the top limit
                                    lim_rev = 1;
                                    t_tmp = threshold-0.01;
                                end
                                if t_tmp<limit(1)&&lim_rev  %cannot find anything that will work
                                    no_deal = 1;
                                end
                                [pivots_tmp,cc_tmp,stk] = find_centroidvstk('dirname',curr_dir,'type',type,'type2',...
                                    type2,'object',object,'threshold',t_tmp,'norm',norm,'zfilter',zfilter,'save',0,'stk',stk);
                                if size(pivots_tmp{1},1)/cdata_tmp(out_idx(m),ref_idx)<ref_cutup  %now within range
                                    disp(['OUTPUT: ',dirout,' - with threshold:',num2str(t_tmp),' with new ratio:',...
                                        num2str(size(pivots_tmp{1},1)/cdata_tmp(out_idx(m)))]);
                                    %actually save out the data
                                    find_centroidvstk('dirname',curr_dir,'type',type,'type2',type2,'object',object,...
                                        'threshold',t_tmp,'norm',norm,'zfilter',zfilter,'dirout',dirout,'stk',stk);
                                    %store info and move on
                                    %check_data = {'experiment','animal','region','channel','orig_thresh','new_thresh','orig_centroid','new_centroid',...
                                    %       'ref_stat','ref_std','upper_bound','lower_bound','in_dir','out_dir'};
                                    add_data = {exp_names{out_idx(m)+1},animal_types{i},regions{j},ch_names{l+1},thresh_tmp{out_idx(m),l},t_tmp,...
                                        cdata_tmp(out_idx(m),l),size(pivots_tmp{1},1),ref_stat,ref_std,ref_cutup,ref_cutdwn,curr_dir,dirout};
                                    check_data = vertcat(check_data,add_data);
                                    break
                                else
                                    disp(['TEST: new threshold:',num2str(t_tmp),' with ratio:',num2str(size(pivots_tmp{1},1)/cdata_tmp(out_idx(m)))]);
                                end
                                if no_deal  %no deal
                                    warning(['Did not find acceptable threshold for ', curr_dir]);
                                    break
                                end
                            end
                        end
                    end
                    %deal with lower bound now
                    out_idx = find(ratio_data(:,l)<=ref_cutdwn);
                    %now fix this centroid
                    if ~isempty(out_idx)  %skip if there are no out of bounds centroids
                        for n = 1:size(out_idx,2)  %go through each centroid
                            %grab the threshold and the locations now
                            threshold = thresh_tmp{out_idx(n),l};
                            loc_cache = loc_tmp{out_idx(n),l};
                            curr_dir = loc_cache{1};
                            dirout = loc_cache{2};
                            stk = [];  %initialize
                            lim_rev = 0;  %a switch to verify that the limit is reached
                            no_deal = 0;
                            disp([curr_dir,' - old threshold:',num2str(threshold),' with ratio:',num2str(ratio_data(out_idx(m),l))]);  %print current directory to command line
                            while 1  %keep on going until we fix it
                                %the centroid here is low.  So we need to
                                %decrease the threshold
                                if isempty(stk)  %first time through and we subtract first
                                    t_tmp = threshold-0.01;
                                else
                                    if lim_rev
                                        t_tmp = t_tmp+0.01;  %add if lim_rev is reached
                                    else
                                        t_tmp = t_tmp-0.01;  %subtract if lim_rev is not reached
                                    end
                                end
                                if t_tmp<limit(1)  %gone past the top limit
                                    lim_rev = 1;
                                    t_tmp = threshold+0.01;
                                end
                                if t_tmp>limit(2)&&lim_rev  %cannot find anything that will work
                                    no_deal = 1;
                                end
                                [pivots_tmp,cc_tmp,stk] = find_centroidvstk('dirname',curr_dir,'type',type,'type2',...
                                    type2,'object',object,'threshold',t_tmp,'norm',norm,'zfilter',zfilter,'save',0,'stk',stk);
                                if size(pivots_tmp{1},1)/cdata_tmp(out_idx(m),ref_idx)>ref_cutdwn  %now within range
                                    disp(['OUTPUT: ',dirout,' - with threshold:',num2str(t_tmp),' with new ratio:',...
                                        num2str(size(pivots_tmp{1},1)/cdata_tmp(out_idx(m)))]);
                                    %actually save out the data
                                    find_centroidvstk('dirname',curr_dir,'type',type,'type2',type2,'object',object,...
                                        'threshold',t_tmp,'norm',norm,'zfilter',zfilter,'dirout',dirout,'stk',stk);
                                    %store info and move on
                                    %check_data = {'experiment','animal','region','channel','orig_thresh','new_thresh','orig_centroid','new_centroid',...
                                    %       'ref_stat','ref_std','upper_bound','lower_bound','in_dir','out_dir'};
                                    add_data = {exp_names{out_idx(n)+1},animal_types{i},regions{j},ch_names{l+1},thresh_tmp{out_idx(n),l},t_tmp,...
                                        cdata_tmp(out_idx(n),l),size(pivots_tmp{1},1),ref_stat,ref_std,ref_cutup,ref_cutdwn,curr_dir,dirout};
                                    check_data = vertcat(check_data,add_data);
                                    break
                                else
                                    disp(['TEST: new threshold:',num2str(t_tmp),' with ratio:',num2str(size(pivots_tmp{1},1)/cdata_tmp(out_idx(m)))]);
                                end
                                if no_deal  %no deal
                                    warning(['Did not find acceptable threshold for ', curr_dir]);
                                    break
                                end
                            end
                        end
                    end
                end
            end
        end
    end
end

%save out data
save([dirname,'check_data',datestr(now,'yyyymmdd.HHMM'),'.mat'],'check_data');

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [centroid_data,method,std_away,dirname] = parse(input)
centroid_data = [];
dirname = [];
filename = [];
method = 'median';
std_away = 2;

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'dirname'
                    dirname = input{1,i+1};
                    %make sure the dirname have a filesep at the end
                    sep_idx = strfind(dirname,filesep);
                    if sep_idx(end)~=size(dirname,2)  %no filesep at end
                        dirname = [dirname,filesep];  %add filesep
                    end
                case 'filename'
                    filename = input{1,i+1};
                case 'method'
                    method = input{1,i+1};
                case 'std_away'
                    std_away = input{1,i+1};
                case 'data'
                    centroid_data = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end

%Get the location of the images you want to open
if isempty(centroid_data)&&isempty(dirname)
    [filename,dirname,filterindex] = uigetfile2({'*.mat','Text files (*.mat)';'*.*','All Files';},...
        'Open Centroid Data File','Multiselect','off');
end

%now get the data
if isempty(centroid_data)
    open([dirname,filename]);
end

%--------------------------------------------------------------------------
%subfunction help conversion of a cell with empty cells to a matrix
function [array_in] = nixemptycell(array_in)
%cell2mat does not work if there is an empty cell
for i = 1:size(array_in,1)
    for j = 1:size(array_in,2)
        if isempty(array_in{i,j})
            array_in{i,j} = nan;  %convert to nan so that later math is simplified
        end
    end
end